/*
 * QrfeProtocollHandlerRFE.cpp
 *
 *  Created on: 15.01.2009
 *      Author: stefan.detter
 */

#include "../inc/reader/QrfeRfeReader.h"

#include "../inc/epc/EPC_Defines.h"

QrfeRfeReader::QrfeRfeReader(QrfeProtocolHandlerRFE* ph, QObject* parent)
	: QrfeReaderInterface(ph, parent)
	, m_ph(ph)
{
	connect(m_ph, SIGNAL(deviceClosed()),
			this,   SLOT(deviceClosed()), Qt::DirectConnection);
	connect(m_ph, SIGNAL(heartBeat()),
			this,   SLOT(heartBeatSlot()), Qt::DirectConnection);
	connect(m_ph, SIGNAL(cyclicInventory(const QrfeGlobal::TagEvent&)),
			this,   SLOT(cyclicInventorySlot(const QrfeGlobal::TagEvent&)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(stateChanged(const int)),
			this,   SLOT(handleProtocolHandlerState(const int)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(statusRegisterChanged(const qulonglong)),
			this,   SLOT(statusRegisterChanged(const qulonglong)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(bootUpFinished(const bool)),
			this,   SLOT(bootUpFinished(const bool)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(notification(uchar,const QByteArray&)),
			this,   SLOT(notificationSlot(uchar,const QByteArray&)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(applicationInterrupt(const QByteArray&)),
			this,   SLOT(applicationInterruptSlot(const QByteArray&)), Qt::DirectConnection);
	connect(m_ph, SIGNAL(gpioValuesChanged (const ulong)),
			this,   SLOT(gpioValuesChanged (const ulong)), Qt::DirectConnection);
}

QrfeRfeReader::~QrfeRfeReader()
{
	// set state to offline
	setState(QrfeGlobal::STATE_OFFLINE);
}


QrfeGlobal::Result QrfeRfeReader::initDevice( )
{
	QrfeGlobal::Result res;

	m_ph->setHeartBeat(HEARTBEAT_OFF);

	res = m_ph->getReaderID ( m_serialNumberExact );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	res = m_ph->getReaderType( m_readerTypeExact );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	m_serialNumber = QrfeGlobal::readerSerialToString(m_serialNumberExact);
	this->traceModuleName = prepareModuleName("ReaderIf-" + QString("%1").arg(m_serialNumberExact, 8, 16, QChar('0')));

	QString type;
	switch( (uchar)(m_readerTypeExact >> 24) )
	{
	case 0x80:
	case 0x81:
		m_readerModule		= QrfeGlobal::VERSION_RFE_PUR_RM1;
		m_readerType		= QrfeGlobal::TYPE_PASSIVE;
		m_tagType			= QrfeGlobal::TAG_GEN2;
		break;
	case 0x82:
		m_readerModule		= QrfeGlobal::VERSION_RFE_PUR_RM2;
		m_readerType		= QrfeGlobal::TYPE_PASSIVE;
		m_tagType			= QrfeGlobal::TAG_GEN2;
		break;

	default:
		m_readerType		= QrfeGlobal::TYPE_PASSIVE;
		break;
	}

	if(type.isNull())
		type = QrfeGlobal::readerModuleToString(m_readerModule);

	if((uchar)(m_readerTypeExact >> 24) != 0xFF)
		m_readerID = type + " - " + m_serialNumber + " @ " + m_ph->deviceName();
	else
		m_readerID = type + " @ " + m_ph->deviceName();

	m_readerTypeExactString = QrfeGlobal::readerTypeToString(m_readerTypeExact);

	res = m_ph->getHardwareRevision( m_hardwareRevision );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	res = m_ph->getSoftwareRevision( m_softwareRevision );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	res = m_ph->getBootloaderRevision( m_bootloaderRevision );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	m_hardwareRevisionString   = QrfeGlobal::hardwareRevisionToString(m_hardwareRevision);
	m_softwareRevisionString   = QrfeGlobal::softwareRevisionToString(m_softwareRevision);
	m_bootloaderRevisionString = QrfeGlobal::bootloaderRevisionToString(m_bootloaderRevision);

	res = m_ph->getCurrentSystem( m_currentSystemString );
	handleProtocolHandlerError(res);
	if(res != QrfeGlobal::RES_OK)
		return res;

	QrfeGlobal::ReaderState state;
	res = m_ph->getCurrentState(state);
	if(res == QrfeGlobal::RES_OK){
		handleProtocolHandlerState(state);
	}

	if(m_ph->getStatusRegister(m_statusRegister) != QrfeGlobal::RES_OK)
		m_statusRegister = 0;

	if(m_ph->getGPIO(m_gpioValues) != QrfeGlobal::RES_OK)
		m_gpioValues = 0;

	if( m_ph->getAntennaCount( m_antennaCount ) != QrfeGlobal::RES_OK)
		m_antennaCount = 1;

	res = m_ph->getDeviceName( m_readerName );
	if(res != QrfeGlobal::RES_OK)
		m_readerName = "";

	res = m_ph->getDeviceLocation( m_readerLocation );
	if(res != QrfeGlobal::RES_OK)
		m_readerName = "";

	if(m_currentSystemString == "Firmware"){
		m_currentSystem = QrfeGlobal::FIRMWARE;
        setHeartBeat(true, 250);
	}
	else
		m_currentSystem = QrfeGlobal::BOOTLOADER;

    if(m_readerType == QrfeGlobal::TYPE_PASSIVE)
    {
        if((m_softwareRevision&0x7FFF) >= 0x0108)
        {
            m_ph->useDeprecatedRfFunctions(false);
        }
    }

	QByteArray temp;
	if(m_ph->getParam(0xFFF0, temp) == QrfeGlobal::RES_OK)
	{
		m_buildDate = QString::fromLatin1(temp, temp.size());
	}
	temp.clear();
	if(m_ph->getParam(0xFFF1, temp) == QrfeGlobal::RES_OK)
	{
		m_buildTime = QString::fromLatin1(temp, temp.size());
	}
	temp.clear();
	if(m_ph->getParam(0xFFF2, temp) == QrfeGlobal::RES_OK)
	{
		m_buildRevision = QString::fromLatin1(temp, temp.size());
	}

	m_deviceInit = true;

	readSettings();

	return QrfeGlobal::RES_OK;
}


QrfeGlobal::Result QrfeRfeReader::setReaderName ( const QString &name )
{
	QrfeGlobal::Result res = m_ph->setDeviceName(name);
	handleProtocolHandlerError(res);

	if(res == QrfeGlobal::RES_OK){
		m_readerName = name;
		emit changedReaderName(name);
	}

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setReaderLocation ( const QString &location )
{
	QrfeGlobal::Result res = m_ph->setDeviceLocation(location);
	handleProtocolHandlerError(res);

	if(res == QrfeGlobal::RES_OK){
		m_readerLocation = location;
		emit changedReaderLocation(location);
	}

	return res;
}


QrfeGlobal::Result QrfeRfeReader::getAttenuation( ushort &maxOutputPower, ushort &currentOutputPower )
{
	QrfeGlobal::Result res = m_ph->getAttenuation(maxOutputPower, currentOutputPower);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getMaxAttenuation( ushort &maxOutputPower )
{
	ushort currentOutputPower;
	QrfeGlobal::Result res = m_ph->getAttenuation(maxOutputPower, currentOutputPower);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getCurrentAttenuation( ushort &currentOutputPower )
{
	ushort maxOutputPower;
	QrfeGlobal::Result res = m_ph->getAttenuation(maxOutputPower, currentOutputPower);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getSensitivity ( short &maxSensitivity, short &minSensitivity, short &currentSensitivity )
{
	QrfeGlobal::Result res = m_ph->getSensitivity(maxSensitivity, minSensitivity, currentSensitivity);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getMaxSensitivity ( short &maxSensitivity )
{
	short minSensitivity, currentSensitivity;
	QrfeGlobal::Result res = m_ph->getSensitivity(maxSensitivity, minSensitivity, currentSensitivity);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getMinSensitivity ( short &minSensitivity )
{
	short maxSensitivity, currentSensitivity;
	QrfeGlobal::Result res = m_ph->getSensitivity(maxSensitivity, minSensitivity, currentSensitivity);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getCurrentSensitivity ( short &currentSensitivity )
{
	short maxSensitivity, minSensitivity;
	QrfeGlobal::Result res = m_ph->getSensitivity(maxSensitivity, minSensitivity, currentSensitivity);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getFrequency ( uchar &mode, uchar &maxFrequencyCount, QList<uint> &frequencies )
{
	QrfeGlobal::Result res = m_ph->getFrequency(mode, maxFrequencyCount, frequencies);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getLbtParams ( ushort &listenTime, ushort &idleTime, ushort &maxAllocTime, short &rssiThreshold )
{
	QrfeGlobal::Result res = m_ph->getLbtParams(listenTime, idleTime, maxAllocTime, rssiThreshold);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setAttenuation( ushort value )
{
	QrfeGlobal::Result res = m_ph->setAttenuation(value);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setFrequency ( uchar mode, QList<uint> frequencies )
{
	QrfeGlobal::Result res = m_ph->setFrequency(mode, frequencies);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setSensitivity ( short targetValue, short &actualValue )
{
	QrfeGlobal::Result res = m_ph->setSensitivity(targetValue, actualValue);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setLbtParams ( ushort listenTime, ushort idleTime, ushort maxAllocTime, short rssiThreshold )
{
	QrfeGlobal::Result res = m_ph->setLbtParams(listenTime, idleTime, maxAllocTime, rssiThreshold);
	handleProtocolHandlerError(res);

	return res;
}



QrfeGlobal::Result QrfeRfeReader::reboot ( )
{
	m_ph->reboot();

	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfeReader::setHeartBeat ( bool on, ushort interval )
{
	QrfeGlobal::Result res;

	res = m_ph->setHeartBeat(HEARTBEAT_DUPLEX_ON, interval);
	m_heartBeatDuplex = true;

	handleProtocolHandlerError(res);

	if(res == QrfeGlobal::RES_OK)
	{
		m_heartBeatInterval = interval;

		m_heartBeatControl->setInterval(m_heartBeatInterval * _heartBeatMultiplier);

		if(on)
			m_heartBeatControl->start();
		else
			m_heartBeatControl->stop();

		m_lastHeartBeat.start();
	}

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setAntennaPower ( bool on )
{
	QrfeGlobal::Result res = m_ph->setAntennaPower(on);
	handleProtocolHandlerError(res);

	return res;
}


QrfeGlobal::Result QrfeRfeReader::switchCommBaudrate ( ulong baudRate )
{
	QrfeGlobal::Result res = m_ph->switchCommBaudrate(baudRate);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::switchCommInterface ( uchar id )
{
	QrfeGlobal::Result res = m_ph->switchCommInterface((eRFE_COMM_INTERFACE_ID)id);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::switchSystem ( )
{
	m_ph->switchSystem();

	//setState(STATE_OFFLINE);

	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfeReader::bootloader ( QByteArray &getMessage, QByteArray setMessage )
{
	QrfeGlobal::Result res = m_ph->bootloader(getMessage, setMessage);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::bootloader ( QByteArray setMessage )
{
    QrfeGlobal::Result res = m_ph->bootloader(setMessage);
    handleProtocolHandlerError(res);

    return res;
}



QrfeGlobal::Result QrfeRfeReader::saveSettingsPermanent ( )
{
	QrfeGlobal::Result res = m_ph->saveSettingsPermanent();
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::restoreFactorySettings ( )
{
	QrfeGlobal::Result res = m_ph->restoreFactorySettings();
	handleProtocolHandlerError(res);

	if(res == QrfeGlobal::RES_OK)
		setHeartBeat(true, 1000);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getParam ( ushort address, QByteArray &value )
{
	QrfeGlobal::Result res = m_ph->getParam(address, value);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setParam ( ushort address, QByteArray value )
{
	QrfeGlobal::Result res = m_ph->setParam(address, value);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setTagType(QrfeGlobal::TagType /*type*/)
{
	return QrfeGlobal::RES_NA;
}


QrfeGlobal::Result QrfeRfeReader::getGPIOCaps ( ulong &mask, ulong &output, ulong &input )
{
	QrfeGlobal::Result res = m_ph->getGPIOCaps(mask, output, input);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getGPIODirection ( ulong &direction ){
	QrfeGlobal::Result res = m_ph->getGPIODirection(direction);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setGPIODirection ( ulong direction )
{
	QrfeGlobal::Result res = m_ph->setGPIODirection(direction);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getGPIO ( ulong &mask )
{
	QrfeGlobal::Result res = m_ph->getGPIO(mask);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setGPIO ( ulong mask )
{
	QrfeGlobal::Result res = m_ph->setGPIO(mask);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::clearGPIO ( ulong mask )
{
	QrfeGlobal::Result res = m_ph->clearGPIO(mask);
	handleProtocolHandlerError(res);

	return res;
}


QrfeGlobal::Result QrfeRfeReader::setAntennaSequence ( const QList<QPair<uchar,ulong> > & sequence)
{
	if( m_antennaCount == 1 )
		return QrfeGlobal::RES_OK;

	QrfeGlobal::Result res = m_ph->setAntennaSequence(sequence);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getAntennaSequence ( QList<QPair<uchar,ulong> > & sequence )
{
	if( m_antennaCount == 1 )
		return QrfeGlobal::RES_OK;

	QrfeGlobal::Result res = m_ph->getAntennaSequence(sequence);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::setWorkingAntenna ( uchar antennaIndex)
{
	if( m_antennaCount == 1 )
		return QrfeGlobal::RES_OK;

	QrfeGlobal::Result res = m_ph->setWorkingAntenna(antennaIndex);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::getWorkingAntenna ( uchar & antennaIndex)
{
	if( m_antennaCount == 1 )
		return QrfeGlobal::RES_OK;

	QrfeGlobal::Result res = m_ph->getWorkingAntenna(antennaIndex);
	handleProtocolHandlerError(res);

	return res;
}


QrfeGlobal::Result QrfeRfeReader::activateNotification ( uchar id )
{
	QrfeGlobal::Result res = m_ph->activateNotification(id);
	handleProtocolHandlerError(res);

	return res;
}

QrfeGlobal::Result QrfeRfeReader::deactivateNotification ( uchar id )
{
	QrfeGlobal::Result res = m_ph->deactivateNotification(id);
	handleProtocolHandlerError(res);

	return res;
}


QrfeGlobal::Result QrfeRfeReader::doSingleInventory ( QList<QrfeGlobal::TagEvent> &epc )
{
	QList<QrfeGlobal::TagEvent> tagList;
	QrfeGlobal::Result res = m_ph->doSingleInventory(tagList);
	handleProtocolHandlerError(res);

	if(res != QrfeGlobal::RES_OK)
		return res;

	epc.clear();
	for(int i = 0; i < tagList.size(); i++)
	{
		QrfeGlobal::TagEvent event = tagList.at(i);
		QrfeGlobal::TagReadEvent ev;
		ev.type 			= m_tagType;
		ev.tagId			= QrfeGlobal::bytesToString(event.tagId);
		ev.tagIdBytes		= event.tagId;

		ev.hasAntenna		= event.hasAntenna;
		if(ev.hasAntenna)
			ev.antennaId 		= event.antennaId;

		ev.hasRSSI			= event.hasRSSI;
		if(ev.hasRSSI)
			ev.rssiPercent		= interpretRSSItoPercent(ev.tagIdBytes, event.rssi);

		ev.hasMemory		= event.hasMemory;
		if(ev.hasMemory)
		{
			ev.memBank			= event.memBank;
			ev.memAddr			= event.memAddr;
			ev.memData			= event.memData;
		}

		ev.hasTrigger		= event.hasTrigger;
		if(ev.hasTrigger)
		{
			ev.triggerSourceId 	= event.trigger;
			ev.triggerSource 	= interpretTriggerId(ev.tagIdBytes, event.trigger);
		}

		ev.hasReadFrequency	= event.hasReadFrequency;
		if(ev.hasReadFrequency)
			ev.readFrequency	= event.readFrequency;

		ev.hasHandle 		= event.hasHandle;
		if(ev.hasHandle)
			ev.handle			= event.handle;

		ev.hasState 		= event.hasState;
		if(ev.hasState)
		{
			ev.state			= event.state;
			ev.stateName 		= interpretStateName(ev.tagIdBytes, event.state);
		}

		ev.hasBattery 		= event.hasBattery;
		if(ev.hasBattery)
			ev.battery			= event.battery;

		ev.hasPC 			= event.hasPC;
		if(ev.hasPC)
			ev.pc				= event.pc;

        ev.hasRN16 			= event.hasRN16;
        if(ev.hasRN16)
            ev.rn16				= event.rn16;

		ev.hasMessageId 	= event.hasMessageID;
		if(ev.hasMessageId)
			ev.messageId		= event.messageID;

	}

	return res;
}

QrfeGlobal::Result QrfeRfeReader::startCyclicInventory ( ulong time )
{
	QrfeGlobal::Result res = m_ph->setCyclicInventory(true, time);
	handleProtocolHandlerError(res);

	if(res != QrfeGlobal::RES_OK)
		return res;

	emit inventoryStarted();

	setAction(QrfeGlobal::ACTION_SCANNING);
	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfeReader::stopCyclicInventory ( )
{
	QrfeGlobal::Result res = m_ph->setCyclicInventory(false);
	handleProtocolHandlerError(res);

	if(res != QrfeGlobal::RES_OK)
		return res;

	emit inventoryStoped();

	setAction(QrfeGlobal::ACTION_IDLE);
	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfeReader::applicationCommand ( const QByteArray& payload )
{
    QrfeGlobal::Result res = m_ph->applicationCommand(payload);
    handleProtocolHandlerError(res);

    return res;
}

QrfeGlobal::Result QrfeRfeReader::applicationCommand ( const QByteArray& payload, QByteArray& response )
{
	QrfeGlobal::Result res = m_ph->applicationCommand(payload, response);
	handleProtocolHandlerError(res);

	return res;
}



void QrfeRfeReader::deviceClosed()
{
	setState(QrfeGlobal::STATE_OFFLINE);
}

void QrfeRfeReader::heartBeatSlot()
{
	uint msecs = m_lastHeartBeat.elapsed();
	m_lastHeartBeat.start();

	m_heartBeatHappened = true;

	setState(QrfeGlobal::STATE_ONLINE);

	trc(4, "[E] Heartbeat - after " + QString::number(msecs) + "ms");

	if(m_heartBeatDuplex)
	{
		m_ph->sendHeartBeat();
	}

	if(filter_heartBeatSlot())
		emit heartBeat();
}

void QrfeRfeReader::cyclicInventorySlot( const QrfeGlobal::TagEvent& event )
{
	trc(3, "[E] CyclicInventory " + tagEventToString(event));

	if(filter_cyclicInventorySlot(event))
	{
		QrfeGlobal::TagReadEvent ev;
		ev.type 			= m_tagType;
		ev.tagId			= QrfeGlobal::bytesToString(event.tagId);
		ev.tagIdBytes		= event.tagId;

		ev.hasAntenna		= event.hasAntenna;
		if(ev.hasAntenna)
			ev.antennaId 		= event.antennaId;

		ev.hasRSSI			= event.hasRSSI;
		if(ev.hasRSSI)
			ev.rssiPercent		= interpretRSSItoPercent(ev.tagIdBytes, event.rssi);

		ev.hasMemory		= event.hasMemory;
		if(ev.hasMemory)
		{
			ev.memBank			= event.memBank;
			ev.memAddr			= event.memAddr;
			ev.memData			= event.memData;
		}

		ev.hasTrigger		= event.hasTrigger;
		if(ev.hasTrigger)
		{
			ev.triggerSourceId 	= event.trigger;
			ev.triggerSource 	= interpretTriggerId(ev.tagIdBytes, event.trigger);
		}

		ev.hasReadFrequency	= event.hasReadFrequency;
		if(ev.hasReadFrequency)
			ev.readFrequency	= event.readFrequency;

		ev.hasHandle 		= event.hasHandle;
		if(ev.hasHandle)
			ev.handle			= event.handle;

		ev.hasState 		= event.hasState;
		if(ev.hasState)
		{
			ev.state			= event.state;
			ev.stateName 		= interpretStateName(ev.tagIdBytes, event.state);
		}

		ev.hasBattery 		= event.hasBattery;
		if(ev.hasBattery)
			ev.battery			= event.battery;

		ev.hasPC 			= event.hasPC;
		if(ev.hasPC)
			ev.pc				= event.pc;

        ev.hasRN16 			= event.hasRN16;
        if(ev.hasRN16)
            ev.rn16				= event.rn16;

		ev.hasMessageId 	= event.hasMessageID;
		if(ev.hasMessageId)
			ev.messageId		= event.messageID;

        ev.hasNxpBrandId	= event.hasNxpBrandId;
        if(ev.hasNxpBrandId)
            ev.nxpBrandId		= event.nxpBrandId;

		ev.hasApplicationInfo 	= event.hasApplicationInfo;
		if(ev.hasApplicationInfo)
			ev.applicationInfo		= event.applicationInfo;


		if(receivers( SIGNAL(exclusive_cyclicInventory(const QrfeGlobal::TagReadEvent&))) > 0)
			emit exclusive_cyclicInventory(ev);
		else
			emit cyclicInventory(ev);
	}
}

void QrfeRfeReader::statusRegisterChanged(const qulonglong statusRegister)
{
	trc(3, "[E] StatusRegisterChanged " + QString("%1").arg(statusRegister, 16, 16, QChar('0')));

	if(filter_statusRegisterChanged(statusRegister))
	{
		m_statusRegister = statusRegister;
		emit changedStatusRegister(statusRegister);
	}
}

void QrfeRfeReader::bootUpFinished(const bool finished)
{
	trc(3, "[E] bootUp " + QString((finished)?"Finished":"Started"));
}

void QrfeRfeReader::notificationSlot(const uchar id, const QByteArray& data)
{
    trc(3, "[E] Notification " + QString("[%1]").arg(id, 2, 16, QChar('0')) + " " + data.toHex());
}

void QrfeRfeReader::applicationInterruptSlot(const QByteArray& data)
{
    trc(3, "[E] Application Interrupt " + data.toHex());

    if(filter_applicationInterruptSlot(data))
	{
    	;
	}
}

void QrfeRfeReader::gpioValuesChanged ( const ulong gpioValues )
{
	trc(3, "[E] GpioValuesChanged " + QString("%1").arg(gpioValues, 8, 16, QChar('0')));

	if(filter_gpioValuesChanged(gpioValues))
	{
		m_gpioValues = gpioValues;
		emit changedGpioValues(gpioValues);
	}
}


bool QrfeRfeReader::filter_heartBeatSlot()
{
	return true;
}

bool QrfeRfeReader::filter_cyclicInventorySlot(const QrfeGlobal::TagEvent& /*event*/)
{
	return true;
}

bool QrfeRfeReader::filter_statusRegisterChanged(const qulonglong /*statusRegister*/)
{
	return true;
}

bool QrfeRfeReader::filter_bootUpFinished(const bool /*finished*/)
{
	return true;
}

bool QrfeRfeReader::filter_notificationSlot(const uchar /*id*/, const QByteArray& /*data*/)
{
	return true;
}

bool QrfeRfeReader::filter_applicationInterruptSlot ( const QByteArray& /*data*/ )
{
	return true;
}

bool QrfeRfeReader::filter_gpioValuesChanged(const ulong /*gpioValues*/)
{
	return true;
}
